<?php
/**
 * AI Assistant for the WPadmin plugin subscription class.
 * This class handles subscription-related functionalities for the plugin.
 *
 * @package AI_Assistant_For_The_WPadmin
 * @since 1.0.0
 */

/**
 * AI Assistant for the WPadmin plugin subscription class.
 * This class handles subscription-related functionalities for the plugin.
 *
 * @package AI_Assistant_For_The_WPadmin
 * @since 1.0.0
 */
class Ai_Assistant_For_Wpadmin_Subscription {
	/**
	 * Constructor method for the class.
	 * Hooks into WordPress actions to handle subscription-related tasks.
	 */
	public static function get_site_url() {
		$site_url = get_site_url();
		$site_url = str_replace( 'http://', '', $site_url );
		$site_url = str_replace( 'https://', '', $site_url );
		$site_url = str_replace( '/', '', $site_url );
		return $site_url;
	}
	/**
	 * Returns the subscription URL for the specified plan type.
	 *
	 * @param string $plan_type The type of subscription plan ('free' or 'premium').
	 * @return string|bool The subscription URL or false if the plan type is invalid.
	 */
	public static function get_subscription_url( $plan_type ) {
		$site_url = self::get_site_url();
		if ( ! in_array( $plan_type, array( 'free', 'premium' ), true ) ) {
			return false; // Invalid plan type.
		}
		$subscription_url = add_query_arg(
			array(
				'subscribe'  => $plan_type,
				'buy_for'    => esc_attr( $site_url ),
				'return_url' => rawurlencode( admin_url( 'admin.php?page=ai-assistant-for-wpadmin-settings' ) ),
			),
			AI_ASSISTANT_MIDDLEWARE
		);
		return $subscription_url;
	}
	/**
	 * Returns the subscription details for the current site.
	 *
	 * @return array An array containing subscription details.
	 */
	public static function get_subscription_details() {
		$site_url         = self::get_site_url();
		$encoded_site_url = base64_encode( $site_url );//phpcs:ignore
		$api_url          = AI_ASSISTANT_MIDDLEWARE . 'wp-json/wpadmin-member/v1/verify-license?site=' . $encoded_site_url;
		$response         = wp_remote_get(
			$api_url,
			array(
				'timeout' => AI_ASSISTANT_TIMEOUT,
				'headers' => array(
					'Content-Type' => 'application/json',
				),
			)
		);
		if ( is_wp_error( $response ) ) {
			$subscription_status = 'error';
			$plan_name           = 'N/A';
			$billing_frequency   = 'N/A';
			$renewal_date        = 'N/A';
			$status              = 'N/A';
			$last_payment        = 'N/A';
		} else {
			$data = json_decode( wp_remote_retrieve_body( $response ), true );
			if ( isset( $data['success'] ) && $data['success'] ) {
				$subscription = $data['subscription'];
				if ( isset( $subscription['status'] ) && ( 'active' === $subscription['status'] || 'pending-cancel' === $subscription['status'] ) ) {
					$subscription_status = 'active' === $subscription['status'] ? 'active' : 'pending-cancel';
					$plan_name           = isset( $subscription['plan_name'] ) ? $subscription['plan_name'] : ( 0.00 === floatval( $subscription['total'] ) ? 'Free' : 'Premium' );
					$billing_frequency   = isset( $subscription['billing_frequency'] ) ? $subscription['billing_frequency'] : 'Monthly';
					$renewal_date        = isset( $subscription['next_payment_date'] ) && ! empty( $subscription['next_payment_date'] ) ? $subscription['next_payment_date'] : 'N/A';
					$status              = 'active' === $subscription['status'] ? 'Active' : 'Active until ' . $subscription['end_date'];
					$last_payment        = $subscription['date_created'];
				} else {
					// Handle expired or no subscription.
					$subscription_status = isset( $subscription['status'] ) ? $subscription['status'] : 'expired';
					$plan_name           = 'N/A';
					$billing_frequency   = 'N/A';
					$renewal_date        = 'N/A';
					$status              = 'N/A';
					$last_payment        = 'N/A';
				}
			} else {
				// Handle error response.
				$subscription_status = 'error';
				$plan_name           = 'N/A';
				$billing_frequency   = 'N/A';
				$renewal_date        = 'N/A';
				$status              = 'N/A';
				$last_payment        = 'N/A';
			}
		}
		return array(
			'subscription_status' => $subscription_status,
			'plan_name'           => $plan_name,
			'billing_frequency'   => $billing_frequency,
			'renewal_date'        => $renewal_date,
			'status'              => $status,
			'last_payment'        => $last_payment,
		);
	}
}
